// Slider functionality
let currentSlide = 0
const slides = document.querySelectorAll('.slide')
const dots = document.querySelectorAll('.dot')
const totalSlides = slides.length

function goToSlide(index) {
  if (slides.length === 0) return
  
  slides[currentSlide].classList.remove('active')
  if (dots[currentSlide]) dots[currentSlide].classList.remove('active')
  
  currentSlide = index
  if (currentSlide >= totalSlides) currentSlide = 0
  if (currentSlide < 0) currentSlide = totalSlides - 1
  
  slides[currentSlide].classList.add('active')
  if (dots[currentSlide]) dots[currentSlide].classList.add('active')
}

function goToNext() {
  goToSlide(currentSlide + 1)
}

function goToPrevious() {
  goToSlide(currentSlide - 1)
}

// Auto-slide
if (slides.length > 0) {
  setInterval(() => {
    goToNext()
  }, 3000)
  
  goToSlide(0)
}

// Applications page
function initIndexPage() {
  loadApplications()
  updateStats()
}

function loadApplications() {
  const saved = localStorage.getItem('courseApplications')
  let applications = []
  
  if (saved) {
    try {
      applications = JSON.parse(saved)
    } catch (e) {
      applications = []
    }
  } else {
    applications = [
      {
        id: 1,
        courseName: 'Основы алгоритмизации и программирования',
        startDate: '2025-05-15',
        status: 'active',
        review: null
      },
      {
        id: 2,
        courseName: 'Основы веб-дизайна',
        startDate: '2025-04-20',
        status: 'completed',
        review: {
          rating: 5,
          text: 'Отличный курс! Очень понравилась подача материала.'
        }
      }
    ]
    localStorage.setItem('courseApplications', JSON.stringify(applications))
  }
  
  renderApplications(applications)
}

function renderApplications(applications) {
  const container = document.getElementById('applications-container')
  if (!container) return
  
  if (applications.length === 0) {
    container.innerHTML = `
      <div class="empty-state-split">
        <div class="empty-icon-wrapper">
          <img src="icons/file-text.svg" alt="File" width="64" height="64">
        </div>
        <h2 class="empty-title-split">Нет заявок</h2>
        <p class="empty-text-split">Создайте первую заявку на курс</p>
        <button class="empty-button-split" onclick="window.location.href='enroll.html'">
          Записаться на курс
        </button>
      </div>
    `
    return
  }
  
  container.innerHTML = applications.map((application) => {
    const date = new Date(application.startDate).toLocaleDateString('ru-RU', { 
      day: 'numeric', 
      month: 'long', 
      year: 'numeric' 
    })
    const reviewSection = application.status === 'completed' 
      ? (application.review 
          ? `
            <div class="review-section-split">
              <div class="review-display-split">
                <div class="review-stars-split">
                  ${Array.from({ length: 5 }).map((_, i) => 
                    `<img src="icons/star.svg" alt="Star" width="20" height="20" style="${i < application.review.rating ? 'filter: brightness(0) saturate(100%) invert(77%) sepia(95%) saturate(1352%) hue-rotate(358deg) brightness(101%) contrast(101%);' : 'opacity: 0.3;'}">`
                  ).join('')}
                </div>
                <p class="review-text-split">${application.review.text}</p>
              </div>
            </div>
          `
          : `
            <div class="review-section-split">
              <div class="review-form-split">
                <div class="review-header-split">
                  <img src="icons/message-square.svg" alt="Message" width="20" height="20">
                  <span>Оставить отзыв</span>
                </div>
                <div class="rating-split" id="rating-${application.id}">
                  ${Array.from({ length: 5 }).map((_, i) => 
                    `<button type="button" class="star-btn-split" onclick="setRating(${application.id}, ${i + 1})">
                      <img src="icons/star.svg" alt="Star" width="24" height="24" id="star-${application.id}-${i + 1}">
                    </button>`
                  ).join('')}
                </div>
                <textarea
                  class="review-textarea-split"
                  id="review-text-${application.id}"
                  placeholder="Напишите ваш отзыв..."
                  rows="3"
                ></textarea>
                <button
                  class="submit-review-split"
                  onclick="submitReview(${application.id})"
                  id="submit-review-${application.id}"
                >
                  Отправить отзыв
                </button>
              </div>
            </div>
          `
        )
      : ''
    
    return `
      <div class="application-item-split ${application.status}">
        <div class="item-header-split">
          <div class="item-number-split">#${application.id}</div>
          <div class="status-indicator-split ${application.status}">
            ${application.status === 'active' 
              ? `<img src="icons/clock.svg" alt="Clock" width="16" height="16"><span>Активна</span>`
              : `<img src="icons/check-circle.svg" alt="Check" width="16" height="16"><span>Завершена</span>`
            }
          </div>
        </div>
        <h3 class="item-title-split">${application.courseName}</h3>
        <div class="item-meta-split">
          <img src="icons/calendar.svg" alt="Calendar" width="18" height="18">
          <span>${date}</span>
        </div>
        ${reviewSection}
      </div>
    `
  }).join('')
}

function updateStats() {
  const saved = localStorage.getItem('courseApplications')
  let applications = []
  
  if (saved) {
    try {
      applications = JSON.parse(saved)
    } catch (e) {
      applications = []
    }
  }
  
  const totalCount = applications.length
  const activeCount = applications.filter(a => a.status === 'active').length
  const completedCount = applications.filter(a => a.status === 'completed').length
  
  const totalEl = document.getElementById('total-count')
  const activeEl = document.getElementById('active-count')
  const completedEl = document.getElementById('completed-count')
  
  if (totalEl) totalEl.textContent = totalCount
  if (activeEl) activeEl.textContent = activeCount
  if (completedEl) completedEl.textContent = completedCount
}

let reviewRatings = {}

function setRating(applicationId, rating) {
  reviewRatings[applicationId] = rating
  
  for (let i = 1; i <= 5; i++) {
    const star = document.getElementById(`star-${applicationId}-${i}`)
    if (star) {
      if (i <= rating) {
        star.style.filter = 'brightness(0) saturate(100%) invert(77%) sepia(95%) saturate(1352%) hue-rotate(358deg) brightness(101%) contrast(101%);'
      } else {
        star.style.filter = 'opacity(0.3)'
      }
    }
  }
}

function submitReview(applicationId) {
  const rating = reviewRatings[applicationId]
  const textarea = document.getElementById(`review-text-${applicationId}`)
  const text = textarea ? textarea.value.trim() : ''
  
  if (!rating || !text) {
    alert('Пожалуйста, выберите рейтинг и напишите отзыв')
    return
  }
  
  const saved = localStorage.getItem('courseApplications')
  let applications = []
  
  if (saved) {
    try {
      applications = JSON.parse(saved)
    } catch (e) {
      applications = []
    }
  }
  
  const updated = applications.map(app => {
    if (app.id === applicationId) {
      return {
        ...app,
        review: {
          rating: rating,
          text: text
        }
      }
    }
    return app
  })
  
  localStorage.setItem('courseApplications', JSON.stringify(updated))
  loadApplications()
  updateStats()
}

// Enroll page
function initEnrollPage() {
  renderCourses()
  setupForm()
  updateSteps()
}

const courses = [
  { 
    id: 'algorithms', 
    name: 'Основы алгоритмизации и программирования', 
    icon: 'code.svg',
    color: '#2563EB',
    description: 'Изучите основы программирования, алгоритмы и структуры данных. Начните свой путь в IT с нуля.',
    duration: '120 часов',
    level: 'Начальный',
    students: '2500+',
    rating: '4.9'
  },
  { 
    id: 'web-design', 
    name: 'Основы веб-дизайна', 
    icon: 'palette.svg',
    color: '#7C3AED',
    description: 'Создавайте красивые и функциональные веб-интерфейсы. Изучите UI/UX дизайн и современные инструменты.',
    duration: '100 часов',
    level: 'Начальный',
    students: '1800+',
    rating: '4.8'
  },
  { 
    id: 'databases', 
    name: 'Основы проектирования баз данных', 
    icon: 'database.svg',
    color: '#10B981',
    description: 'Освойте проектирование и работу с базами данных. SQL, нормализация, оптимизация запросов.',
    duration: '140 часов',
    level: 'Средний',
    students: '1200+',
    rating: '4.9'
  }
]

let selectedCourse = ''

function renderCourses() {
  const container = document.getElementById('courses-container')
  if (!container) return
  
  container.innerHTML = courses.map((course) => {
    return `
      <button
        type="button"
        class="course-card-split"
        data-course="${course.id}"
        onclick="selectCourse('${course.id}')"
      >
        <div class="course-card-header-split">
          <div class="course-icon-split" style="background: ${course.color}15; color: ${course.color};">
            <img src="icons/${course.icon}" alt="${course.name}" width="32" height="32">
          </div>
        </div>
        <h3 class="course-name-split">${course.name}</h3>
        <p class="course-desc-split">${course.description}</p>
        <div class="course-stats-split">
          <div class="course-stat-split">
            <img src="icons/clock.svg" alt="Clock" width="16" height="16">
            <span>${course.duration}</span>
          </div>
          <div class="course-stat-split">
            <img src="icons/users.svg" alt="Users" width="16" height="16">
            <span>${course.students}</span>
          </div>
          <div class="course-stat-split">
            <img src="icons/award.svg" alt="Award" width="16" height="16">
            <span>${course.rating}</span>
          </div>
        </div>
        <div class="course-level-split">
          <span>${course.level}</span>
        </div>
      </button>
    `
  }).join('')
}

function selectCourse(courseId) {
  selectedCourse = courseId
  
  document.querySelectorAll('.course-card-split').forEach(btn => {
    btn.classList.remove('selected')
    if (btn.dataset.course === courseId) {
      btn.classList.add('selected')
      const checkBadge = document.createElement('div')
      checkBadge.className = 'check-badge-split'
      checkBadge.innerHTML = `<img src="icons/check-circle.svg" alt="Check" width="24" height="24">`
      btn.querySelector('.course-card-header-split').appendChild(checkBadge)
    } else {
      const checkBadge = btn.querySelector('.check-badge-split')
      if (checkBadge) checkBadge.remove()
    }
  })
  
  const courseData = courses.find(c => c.id === courseId)
  if (courseData) {
    showCoursePreview(courseData)
  }
  
  const dateSection = document.getElementById('date-section')
  if (dateSection) {
    dateSection.style.display = 'block'
  }
  
  updateSteps()
  updateSubmitButton()
}

function showCoursePreview(courseData) {
  const preview = document.getElementById('course-preview')
  if (!preview) return
  
  preview.style.display = 'block'
  document.getElementById('preview-icon').style.background = `${courseData.color}15`
  document.getElementById('preview-icon-img').src = `icons/${courseData.icon}`
  document.getElementById('preview-title').textContent = courseData.name
  document.getElementById('preview-meta').innerHTML = `
    <div class="preview-meta-item">
      <img src="icons/clock.svg" alt="Clock" width="16" height="16">
      <span>${courseData.duration}</span>
    </div>
    <div class="preview-meta-item">
      <img src="icons/users.svg" alt="Users" width="16" height="16">
      <span>${courseData.students}</span>
    </div>
  `
}

function updateSteps() {
  const step1 = document.getElementById('step-1')
  const step2 = document.getElementById('step-2')
  const step1Status = document.getElementById('step-1-status')
  const step2Status = document.getElementById('step-2-status')
  const dateInput = document.getElementById('date-input')
  
  if (selectedCourse) {
    if (step1) step1.classList.add('active', 'completed')
    if (step1Status) step1Status.textContent = 'Выбрано'
    
    if (dateInput && dateInput.value) {
      if (step2) step2.classList.add('active', 'completed')
      if (step2Status) step2Status.textContent = 'Выбрано'
    } else {
      if (step2) {
        step2.classList.add('active')
        step2.classList.remove('completed')
      }
      if (step2Status) step2Status.textContent = 'Не выбрано'
    }
  } else {
    if (step1) {
      step1.classList.remove('active', 'completed')
    }
    if (step1Status) step1Status.textContent = 'Не выбрано'
  }
}

function setupForm() {
  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]
  
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    dateInput.min = minDateStr
    dateInput.max = maxDateStr
    dateInput.addEventListener('change', () => {
      updateSteps()
      updateSubmitButton()
    })
  }
  
  const form = document.getElementById('enroll-form')
  if (form) {
    form.addEventListener('submit', handleSubmit)
  }
}

function updateSubmitButton() {
  const dateInput = document.getElementById('date-input')
  const submitBtn = document.getElementById('submit-btn')
  
  if (submitBtn && dateInput) {
    submitBtn.disabled = !selectedCourse || !dateInput.value
  }
}

function handleSubmit(e) {
  e.preventDefault()
  
  if (!selectedCourse) {
    alert('Пожалуйста, выберите курс')
    return
  }
  
  const dateInput = document.getElementById('date-input')
  if (!dateInput || !dateInput.value) {
    alert('Пожалуйста, выберите дату')
    return
  }
  
  const saved = localStorage.getItem('courseApplications')
  const applications = saved ? JSON.parse(saved) : []
  
  const courseData = courses.find(c => c.id === selectedCourse)
  
  const newApplication = {
    id: Date.now(),
    courseName: courseData.name,
    startDate: dateInput.value,
    status: 'active',
    review: null
  }
  
  applications.push(newApplication)
  localStorage.setItem('courseApplications', JSON.stringify(applications))
  
  const successMsg = document.getElementById('success-message')
  if (successMsg) {
    successMsg.style.display = 'flex'
  }
  
  setTimeout(() => {
    window.location.href = 'index.html'
  }, 2000)
}

// Initialize page
function initPage() {
  if (window.location.pathname.includes('enroll.html')) {
    initEnrollPage()
  } else {
    initIndexPage()
  }
}

// Run on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initPage)
} else {
  initPage()
}

